/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.serializer.memory;

import java.lang.ref.WeakReference;
import java.nio.ByteBuffer;
import org.eclipse.serializer.exceptions.BufferRegistryException;
import org.eclipse.serializer.hashing.XHashing;
import org.eclipse.serializer.math.XMath;

public class BufferRegistry {
    private final int maximumCapacityBound;
    private int hashRange;
    private int increaseBound;
    private int shrinkBound;
    private int currentLowestFreeIndex;
    private int size;
    private Entry[] hashTable;
    private Entry[] indexTable;
    private long hollowEncounters;

    BufferRegistry(int maximumCapacityBound) {
        this(maximumCapacityBound, 1);
    }

    BufferRegistry(int maximumCapacityBound, int initialCapacityBound) {
        this.maximumCapacityBound = maximumCapacityBound;
        this.createArrays(XHashing.padHashLength(initialCapacityBound));
    }

    private void createArrays(int capacity) {
        this.updateState(new Entry[capacity], new Entry[capacity], 0);
    }

    private void updateState(Entry[] hashTable, Entry[] indexTable, int size) {
        this.hashTable = hashTable;
        this.indexTable = indexTable;
        this.increaseBound = hashTable.length;
        this.shrinkBound = hashTable.length / 2;
        this.hashRange = hashTable.length - 1;
        this.size = size;
        this.updateLowestFreeIndex();
    }

    private void updateLowestFreeIndex() {
        this.currentLowestFreeIndex = BufferRegistry.determineLowestFreeIndex(this.indexTable);
    }

    private int hash(ByteBuffer byteBuffer) {
        return System.identityHashCode(byteBuffer) & this.hashRange;
    }

    private void updateCurrentLowestFreeIndex(int freeIndex) {
        if (freeIndex >= this.currentLowestFreeIndex) {
            return;
        }
        this.currentLowestFreeIndex = freeIndex;
    }

    private void resetHollowEncounters() {
        this.hollowEncounters = 0L;
    }

    private boolean hasHollowEncounters() {
        return this.hollowEncounters > 0L;
    }

    final int ensureRegistered(ByteBuffer byteBuffer) {
        Entry e = this.hashTable[this.hash(byteBuffer)];
        while (e != null) {
            if (e.get() == byteBuffer) {
                return e.index;
            }
            if (e.isHollow()) {
                ++this.hollowEncounters;
            }
            e = e.link;
        }
        if (++this.size >= this.increaseBound) {
            this.checkForIncrementingRebuild();
        }
        int index = this.determineFreeIndex();
        Entry entry = new Entry(byteBuffer, index, this.hashTable[this.hash(byteBuffer)]);
        this.hashTable[this.hash((ByteBuffer)byteBuffer)] = entry;
        this.indexTable[index] = entry;
        return index;
    }

    final ByteBuffer lookupBuffer(int index) {
        return this.indexTable[index] != null ? (ByteBuffer)this.indexTable[index].get() : null;
    }

    private void checkForIncrementingRebuild() {
        if (this.hasHollowEncounters()) {
            this.cleanUp();
        } else {
            this.incrementingRebuild();
        }
    }

    private void incrementingRebuild() {
        if (this.increaseBound >= this.maximumCapacityBound) {
            --this.size;
            throw new BufferRegistryException("Buffer registry cannot be increased beyond the specified maximum capacity of " + this.maximumCapacityBound);
        }
        if (this.increaseBound >= XMath.highestPowerOf2_int()) {
            --this.size;
            throw new BufferRegistryException("Buffer registry cannot be increased beyond the technical maximum capacity of " + XMath.highestPowerOf2_int());
        }
        this.rebuild(this.increaseBound * 2);
    }

    private void cleanUp() {
        Entry[] indexTable = this.indexTable;
        Entry[] hashTable = this.hashTable;
        int capacity = hashTable.length;
        int size = this.size;
        int i = 0;
        while (i < capacity) {
            Entry e = hashTable[i];
            Entry last = null;
            while (e != null) {
                if (e.isHollow()) {
                    indexTable[e.index] = null;
                    if (last == null) {
                        hashTable[i] = e.link;
                    } else {
                        last.link = e.link;
                    }
                    --size;
                } else {
                    last = e;
                }
                e = e.link;
            }
            ++i;
        }
        this.updateLowestFreeIndex();
        this.size = size;
        this.resetHollowEncounters();
    }

    private static int determineLowestFreeIndex(Entry[] indexTable) {
        int i = 0;
        while (i < indexTable.length) {
            if (indexTable[i] == null) {
                return i;
            }
            ++i;
        }
        throw new Error();
    }

    private void shrink() {
        this.rebuild(this.shrinkBound);
    }

    private void rebuild(int newCapacity) {
        Entry[] oldHashTable = this.hashTable;
        int oldCapacity = oldHashTable.length;
        int newHashRange = newCapacity - 1;
        Entry[] newIndxTable = new Entry[newCapacity];
        Entry[] newHashTable = new Entry[newCapacity];
        int size = this.size;
        int i = 0;
        while (i < oldCapacity) {
            Entry e = oldHashTable[i];
            while (e != null) {
                ByteBuffer bb = (ByteBuffer)e.get();
                if (bb == null) {
                    --size;
                } else {
                    Entry entry = new Entry(bb, e.index, newHashTable[System.identityHashCode(bb) & newHashRange]);
                    newHashTable[System.identityHashCode((Object)bb) & newHashRange] = entry;
                    newIndxTable[e.index] = entry;
                }
                e = e.link;
            }
            ++i;
        }
        this.updateState(newHashTable, newIndxTable, size);
        this.resetHollowEncounters();
    }

    private int determineFreeIndex() {
        int currentFreeIndex = this.currentLowestFreeIndex;
        Entry[] indexTable = this.indexTable;
        int i = currentFreeIndex + 1;
        while (i < indexTable.length) {
            if (indexTable[i] == null) {
                this.currentLowestFreeIndex = i;
                return currentFreeIndex;
            }
            ++i;
        }
        throw new Error("Inconsistent byte buffer registry: currentLowestFreeIndex = " + this.currentLowestFreeIndex + ", indexTable.length = " + this.indexTable.length + "." + " No free index found." + " Size = " + this.size);
    }

    private void removeEntry(ByteBuffer byteBuffer, Entry entry, Entry last) {
        if (last == null) {
            this.hashTable[System.identityHashCode((Object)byteBuffer) & this.hashRange] = entry.link;
        } else {
            last.link = entry.link;
        }
        this.clearIndex(entry.index);
        if (--this.size < this.shrinkBound) {
            this.checkShrink();
        }
    }

    private void checkShrink() {
        Entry[] indexTable = this.indexTable;
        int shrinkBound = this.shrinkBound;
        int i = this.increaseBound;
        while (--i >= shrinkBound) {
            if (indexTable[i] == null) continue;
            return;
        }
        this.shrink();
    }

    private void clearIndex(int index) {
        this.indexTable[index] = null;
        this.updateCurrentLowestFreeIndex(index);
    }

    final ByteBuffer ensureRemoved(ByteBuffer byteBuffer) {
        Entry e = this.hashTable[this.hash(byteBuffer)];
        Entry last = null;
        while (e != null) {
            if (e.get() == byteBuffer) {
                this.removeEntry(byteBuffer, e, last);
                return byteBuffer;
            }
            if (e.isHollow()) {
                ++this.hollowEncounters;
            }
            last = e;
            e = last.link;
        }
        return null;
    }

    static final class Entry
    extends WeakReference<ByteBuffer> {
        final int index;
        Entry link;

        Entry(ByteBuffer referent, int index, Entry nextHashed) {
            super(referent);
            this.index = index;
            this.link = nextHashed;
        }

        final boolean isHollow() {
            return this.get() == null;
        }
    }
}

