/*
 * Copyright (c) 2009-2010, EzWare
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.Redistributions
 * in binary form must reproduce the above copyright notice, this list of
 * conditions and the following disclaimer in the documentation and/or
 * other materials provided with the distribution.Neither the name of the
 * EzWare nor the names of its contributors may be used to endorse or
 * promote products derived from this software without specific prior
 * written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 */

package com.ezware.dialog.task;

import java.awt.Component;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.KeyboardFocusManager;
import java.awt.Window;
import java.awt.Dialog.ModalityType;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.ListIterator;
import java.util.Locale;
import java.util.MissingResourceException;
import java.util.ResourceBundle;
import java.util.Set;

import javax.swing.Icon;
import javax.swing.JComponent;
import javax.swing.JDialog;
import javax.swing.SwingUtilities;
import javax.swing.UIManager;

import com.ezware.common.EmptyIcon;
import com.ezware.common.Strings;
import com.ezware.common.SwingBean;
import com.ezware.dialog.task.design.TaskDialogContent;
public class TaskDialog extends SwingBean {

	private static final String INSTANCE_PROPERTY = "task.dialog.instance";

	private static final String DEBUG_PROPERTY    = "task.dialog.debug";

	static final String I18N_PREFIX = "@@";
	private static final String LOCALE_BUNDLE_NAME = "task-dialog";

	private static IContentDesign design = ContentDesignFactory.getDesignByOperatingSystem();

	static {
		getDesign().updateUIDefaults();
	}


	static final IContentDesign getDesign() {
		return design;
	}

	/**
	 * Makes resource bundle key based on the provided text
	 * @param text
	 * @return
	 */
	public static final String makeKey( String text ) {
		return I18N_PREFIX + text;
	}

	/**
	 * Returns task dialog instance associated with source component
	 * @param source
	 * @return task dialog instance associated with source component or null if none
	 */
	public final static TaskDialog getInstance( Component source ) {

		if (  source instanceof Component ) {
			Window w = SwingUtilities.getWindowAncestor(source);
			if ( w instanceof JDialog ) {
				JComponent c = (JComponent) ((JDialog)w).getContentPane();
				return (TaskDialog) c.getClientProperty(INSTANCE_PROPERTY);
			}
		}

		return null;

	}

	/**
	 * Sets debug mode for task dialog framework.
	 * @param debug
	 */
	public final static void setDebugMode( boolean debug ) {

		if ( debug ) {
			System.setProperty(DEBUG_PROPERTY, "true");
		} else {
			System.clearProperty(DEBUG_PROPERTY);
		}

	}

	/**
	 * True if debug mode is set. In debug mode component bounds and grid cells are outlined.
	 * This helps with debugging overall appearance of task dialog UI.
	 * @return true if debug mode is set
	 */
	public final static boolean isDebugMode() {
		return System.getProperty(DEBUG_PROPERTY) != null;
	}


	/**
	 * Set of standard dialog icons. Look and Feel dependent
	 */
	public static enum StandardIcon implements Icon {

		INFO    ( "OptionPane.informationIcon"),
		QUESTION( "OptionPane.questionIcon"   ),
		WARNING ( "OptionPane.warningIcon"    ),
		ERROR   ( "OptionPane.errorIcon"      );

		private final String key;

		private StandardIcon( String key ) {
			this.key = key;
		}

		@Override
		public int getIconHeight() {
			return getIcon().getIconHeight();
		}

		@Override
		public int getIconWidth() {
			return getIcon().getIconWidth();
		}

		@Override
		public void paintIcon(Component c, Graphics g, int x, int y) {
			getIcon().paintIcon(c, g, x, y);
		}


		/**
		 * Always returns an valid instance of icon.
		 * @return
		 */
		private synchronized javax.swing.Icon getIcon() {
			Icon icon = UIManager.getIcon(key); // No caching to facilitate LAF switching.
			return icon == null? getEmptyIcon(): icon;
		}

		private Icon emptyIcon;

		//TODO: Need to use default icons in case they don't exist (Linux GTK)
		private synchronized Icon getEmptyIcon() {
			if (emptyIcon == null) emptyIcon = EmptyIcon.hidden();
			return emptyIcon;
		}

	}


	/**
	 * Standard Command Tags, correspond directly to button tags in MigLayout
	 *
	 */
	public static enum CommandTag {

		OK("ok", true),         // OK button
		CANCEL("cancel"),       // Cancel button
		HELP("help"),           // Help button that is normally on the right
		HELP2("help2", "Help"),         // Help button that on some platforms are placed on the left
		YES("yes", true),       // Yes button
		NO("yes"),              // No button
		APPLY("apply"),         // Apply button
		NEXT("next", true),     // Next or Forward Button
		BACK("back"),           // Previous or Back Button
		FINISH("finish", true), // Finish button
		LEFT("left"),       // Button that should normally always be placed on the far left
		RIGHT("right");     // Button that should normally always be placed on the far right


		private String tag;
		private final String defaultTitle;
		private boolean useValidationResult = false;

		CommandTag( String tag, String defaultTitle, boolean useValidationResult ) {
			this.tag = "tag " + tag;
			this.defaultTitle =  Strings.isEmpty(defaultTitle)? Strings.capitalize(tag) : makeKey(defaultTitle);
			this.useValidationResult = useValidationResult;
		}

		CommandTag( String tag, String defaultTitle ) {
			this( tag, defaultTitle, false );
		}

		CommandTag( String tag ) {
			this( tag, null, false );
		}

		CommandTag( String tag, boolean useValidationResult ) {
			this( tag, null, useValidationResult );
		}

		public String getDefaultTitle() {
			return defaultTitle;
		}
		@Override
		public String toString() {
			return tag;
		}

		public boolean isEnabled( boolean validationResult ) {
			return useValidationResult? validationResult: true;
		}


	}

	public interface ValidationListener {

		void validationFinished( boolean validationResult ); // possibly need a collection of errors/warnings

	}

	/**
	 * Interface to define task dialog commands
	 *
	 */
	public interface Command {

		public String getTitle();
		public CommandTag getTag();
		public String getDescription();
		public boolean isClosing();
		public int getWaitInterval(); // in seconds
		public boolean isEnabled( boolean validationResult );
	}


	/**
	 * Set of standard task dialog commands
	 */
	public static enum StandardCommand implements Command {

		OK    ( CommandTag.OK  ),
		CANCEL( CommandTag.CANCEL );

		private final CommandTag tag;
		private final boolean closing;

		StandardCommand( CommandTag tag, boolean closing ) {
			this.tag     = tag;
			this.closing = closing;
		}


		StandardCommand( CommandTag tag ) {
			this( tag, true );
		}

		@Override
		public String getDescription() {
			return null;
		}

		@Override
		public CommandTag getTag() {
			return tag;
		}

		@Override
		public String getTitle() {
			return tag.getDefaultTitle();
		}


		@Override
		public boolean isClosing() {
			return closing;
		}

		@Override
		public int getWaitInterval() {
			return 0;
		}

		public boolean isEnabled( boolean validationResult ) {
			return tag.isEnabled(validationResult);
		}

		/**
		 * Creates a copy of the command with specified title
		 * @param title new title. Used as key for i18n if starts with I18N_PREFIX
		 * @param waitInterval in seconds
		 * @return copy of the command
		 */
		public Command derive( final String title, final int waitInterval ) {

			return new CustomCommand( StandardCommand.this ) {

				@Override
				public String getTitle() {
					return title;
				}

				@Override
				public int getWaitInterval() {
					return waitInterval;
				}

			};

		}

		/**
		 * Creates a copy of the command with specified title
		 * @param title new title. Used as key for i18n if starts with #I18N_PREFIX
		 * @return copy of the command
		 */
		public Command derive( final String title ) {

			return new CustomCommand( StandardCommand.this ) {

				@Override
				public String getTitle() {
					return title;
				}

			};

		}

	}


	public static abstract class CustomCommand implements Command {

		private final StandardCommand command;

		public CustomCommand( StandardCommand command ) {
			if ( command == null ) throw new IllegalArgumentException("Command should not be null");
			this.command = command;
		}

		@Override
		public String getDescription() {
			return command.getDescription();
		}

		@Override
		public CommandTag getTag() {
			return command.getTag();
		}

		@Override
		public String getTitle() {
			return command.getTitle();
		}

		@Override
		public boolean isClosing() {
			return command.isClosing();
		}

		@Override
		public int getWaitInterval() {
			return command.getWaitInterval();
		}

		public boolean isEnabled( boolean validationResult ) {
			return command.isEnabled(validationResult);
		}

		@Override
		public boolean equals(Object obj) {
			return command.equals(obj);
		}


	}

	/**
	 * Task Dialog Details
	 *
	 */
	public interface Details {

		 /**
		  * Return text for collapsed label
		  * @return
		  */
		 String getCollapsedLabel();

		 /**
		  * Sets text for collapsed label
		  * @param label
		  */
		 void setCollapsedLabel( String label );

		 /**
		  * Returns text for expanded label
		  * @return
		  */
		 String getExpandedLabel();

		 /**
		  * Sets text for expanded label
		  * @param label
		  */
		 void setExpandedLabel( String label );

		 /**
		  * Checks if details are in expansion state
		  * @return
		  */
		 boolean isExpanded();

		 /**
		  * Sets expansion state
		  * @param expanded
		  */
		 void setExpanded( boolean expanded );

		 /**
		  * Returns component which becomes visible when details are expanded
		  * @return
		  */
		 JComponent getExpandableComponent();

		 /**
		  * Sets component which becomes visible when details are expanded
		  * @param cmpt
		  */
		 void setExpandableComponent( JComponent cmpt );

	}


	/**
	 *	Task Dialog Footer
	 *
	 */
	public interface Footer {


		/**
		 * True if footer's check box is selected (checked)
		 * @return
		 */
		boolean isCheckBoxSelected();

		/**
		 * Sets footer's check box selection status
		 * @param selected
		 */
		void setCheckBoxSelected( boolean selected );


		/**
		 * Returns footer's check box text
		 * @return
		 */
		String getCheckBoxText();


		/**
		 * Sets footer's check box text. Check box is only visible if it has a text
		 * @param text
		 */
		void setCheckBoxText( String text );

		/**
		 * Returns footer's text icon
		 * @return
		 */
		Icon getIcon();


		/**
		 * Sets footer's text icon. Icon is only visible if corresponding text is not empty
		 * @param icon
		 */
		void setIcon( Icon icon );


		/**
		 * Returns footer's text
		 * @return
		 */
		String getText();


		/**
		 * Sets footer's text. The text and corresponding icon are visible if text is not empty
		 * @param text
		 */
		void setText(String text);


	}


  /*----------------------------------------------------------------------------------------------*/

	private Locale resourceBundleLocale = null; // has to be null
	private ResourceBundle resourceBundle = null;

	private Command result = StandardCommand.CANCEL;

	private final JDialog dlg;
	private final TaskDialogContent content;

	private Set<Command> commands = new LinkedHashSet<Command>( Arrays.asList(StandardCommand.OK));
	private final List<ValidationListener> validationListeners = new ArrayList<ValidationListener>();

	/**
	 * Creates a task dialog.<br/>
	 * Automatically pick up currently active window as a parent window
	 * @param title
	 */
	public TaskDialog( String title ) {

		dlg = new JDialog( KeyboardFocusManager.getCurrentKeyboardFocusManager().getActiveWindow() );

		dlg.setMinimumSize( new Dimension(300,150)); //TODO: make constants - should be based on LAF
		setResizable(false);
		setModalityType(JDialog.DEFAULT_MODALITY_TYPE); //TODO explore different modality types

		dlg.addWindowListener( new WindowAdapter() {
			@Override
			public void windowClosing(WindowEvent e) {
				result = StandardCommand.CANCEL;
			}
		});

		setTitle(title);
		content = design.buildContent();
		dlg.setContentPane( content );
		//TaskDialogTools.setInstance(dlg, this);
		// store task dialog instance
		JComponent c = (JComponent) dlg.getContentPane();
		c.putClientProperty(INSTANCE_PROPERTY, this);

	}

	public ModalityType getModalityType() {
		return dlg.getModalityType();
	}

	public void setModalityType( ModalityType modalityType ) {
		dlg.setModalityType(modalityType);
	}

	public boolean isResizable() {
		return dlg.isResizable();
	}

	public void setResizable( boolean resizable ) {
		dlg.setResizable(resizable);
	}

	public final void addValidationListener( ValidationListener listener ) {
		if ( listener != null ) {
			validationListeners.add(listener);
		}
	}

	public final void removeValidationListener( ValidationListener listener ) {
		if ( listener != null ) {
			validationListeners.remove(listener);
		}
	}

	public final void fireValidationFinished( boolean validationResult ) {

		// Iterate in reverse sequence so 'newer' listeners get message first
		ListIterator<ValidationListener> iter = validationListeners.listIterator();
	    while (iter.hasPrevious()) {
	    	iter.previous().validationFinished(validationResult);
	    }
	}


	/**
	 * Gets the Locale object that is associated with this window, if the locale has been set.
	 * If no locale has been set, then the default locale is returned.
	 * @return the locale that is set for this dialog
	 */
	public Locale getLocale() {
		return dlg.getLocale();
	}

	/**
	 * Sets locale which will be used as dialog's locale
	 * @param newLocale null is allowed and will be interpreted as default locale
	 */
	public void setLocale( final Locale locale ) {
		dlg.setLocale(locale);
	}

	private synchronized final ResourceBundle getLocaleBundle() {

		Locale currentLocale = getLocale();
		if ( !currentLocale.equals(resourceBundleLocale)) {
			resourceBundleLocale = currentLocale;
			resourceBundle = ResourceBundle.getBundle(
				LOCALE_BUNDLE_NAME,
				resourceBundleLocale,
				getClass().getClassLoader() );
		}
		return resourceBundle;

	}


	/**
	 * Returns a string localized using currently set locale
	 * @param key
	 * @return
	 */
	public String getLocalizedString( String key ) {
		try {
			return getLocaleBundle().getString(key);
		} catch ( MissingResourceException ex ) {
			return String.format("<%s>", key);
		}
	}

	/**
	 * Tries to use text as a key to get localized text. If not found the text itself is returned
	 * @param text
	 * @return
	 */
	public String getString( String text ) {
		return text.startsWith(I18N_PREFIX)? getLocalizedString(text.substring(I18N_PREFIX.length())) : text;
	}


	/**
	 * Shows or hides this {@code Dialog} depending on the value of parameter
	 * @param visible if true dialog is shown
	 */
	public void setVisible( boolean visible ) {

		if ( visible ) {
			// set commands first cause they may depend on "visible" property change
			content.setCommands(commands, getDesign().isCommandButtonSizeLocked());
		}

		if ( !firePropertyChange("visible", isVisible(), visible)) return;

		if ( visible ) {
			dlg.pack();

			// location is set relative to currently active window
			// this way task dialog stays on the same monitor as it's owner
			dlg.setLocationRelativeTo( KeyboardFocusManager.getCurrentKeyboardFocusManager().getActiveWindow() );
		}
		dlg.setVisible(visible);
	}

	/**
	 * Determines whether this component should be visible when its
     * parent is visible.
	 * @return true if visible
	 */
	public boolean isVisible() {
		return dlg.isVisible();
	}

	public Command getResult() {
		return result;
	}

	public void setResult( Command result ) {
		this.result = result; // should always set
		firePropertyChange("result", null, result);
	}

	/**
	 * Shows the dialog
	 * @return dialog result
	 */
	public Command show() {
		setVisible(true);
		return getResult();
	}

	@Override
	public String toString() {
		return getTitle();
	}


	/**
	 * Returns the title of the dialog
	 * @return
	 */
	public String getTitle() {
		return dlg.getTitle();
	}

	/**
	 * Sets the title of the dialog
	 * @param title
	 */
	public void setTitle( String title ) {
		dlg.setTitle( getString(title));
	}

	/**
	 * Sets icon for the dialog
	 * @param icon
	 */
	public void setIcon( Icon icon) {
		if ( firePropertyChange("icon", getIcon(), icon ))  {
			content.setMainIcon(icon);
		}
	}

	/**
	 * Returns dialog's icon
	 * @return
	 */
	public Icon getIcon() {
		return content.getMainIcon();
	}

	/**
	 * Sets dialog's instruction
	 * @param instruction
	 */
	public void setInstruction( String instruction ) {
		if ( firePropertyChange("instruction", getInstruction(), instruction )) {
			content.setInstruction(instruction);
		}
	}

	/**
	 * Returns dialog instruction
	 * @return
	 */
	public String getInstruction() {
		return content.getInstruction();
	}

	/**
	 * Sets dialog text
	 * @param text
	 */
	public void setText( String text ) {
		if ( firePropertyChange("text", getText(), text )) {
			content.setMainText(text);
		}
	}

	/**
	 * Returns Dialog text
	 * @return
	 */
	public String getText() {
		return content.getMainText();
	}

	/**
	 * Sets Dialog component
	 * @param c
	 */
	public void setFixedComponent( JComponent c ) {
		content.setComponent(c);
	}

	/**
	 * Returns dialog's component
	 * @return
	 */
	public JComponent getFixedComponent() {
		return content.getComponent();
	}


	public TaskDialog.Details getDetails() {
		return content;
	}

	public TaskDialog.Footer getFooter() {
		return content;
	}


	public void setCommands(Collection<TaskDialog.Command> commands) {
		this.commands = new LinkedHashSet<Command>(commands);
	}

	public void setCommands( TaskDialog.Command... commands) {
		setCommands( Arrays.asList(commands));
	}


	public Collection<TaskDialog.Command> getCommands() {
		return commands;
	}

	public boolean isCommandsVisible() {
		return content.isCommandsVisible();
	}

	public void setCommandsVisible( boolean visible ) {
		content.setCommandsVisible(visible);
	}

}

