package com.schneide.internal.anttask.testcheck;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.util.ArrayList;
import java.util.List;
import java.util.StringTokenizer;

import org.apache.tools.ant.DirectoryScanner;
import org.apache.tools.ant.Project;
import org.apache.tools.ant.types.FileSet;

import com.schneide.internal.anttask.helper.AbstractAntTask;

public class CheckTestPackageTask extends AbstractAntTask {

    private final List<FileSet> testJavaFiles;
    private String encoding;

    public CheckTestPackageTask() {
        super();
        this.testJavaFiles = new ArrayList<FileSet>();
        this.encoding = System.getProperty("file.encoding"); //$NON-NLS-1$
    }

    public void addConfigured(FileSet targetFileSet) {
        this.testJavaFiles.add(targetFileSet);
    }

    public void setEncoding(String newEncoding) {
        this.encoding = newEncoding;
    }

    protected Iterable<File> collectFiles() {
        List<File> result = new ArrayList<File>();
        for (FileSet fileset : this.testJavaFiles) {
            DirectoryScanner scanner = fileset.getDirectoryScanner(getProject());
            String[] files = scanner.getIncludedFiles();
            for (String string : files) {
                result.add(new File(scanner.getBasedir(), string));
            }
        }
        return result;
    }

    @Override
    protected void executeHandled() throws Exception {
        Iterable<File> testFiles = collectFiles();
        for (File testFile : testFiles) {
            examine(new FiletedClassFile(testFile, this.encoding));
        }
    }

    protected void examine(FiletedClassFile testFile) throws IOException {
        for (String importedClass : testFile.getImportedClasses()) {
            String justClassName = getShortClassNameFrom(importedClass);
            String testName = testFile.getClassName();
            log("Checking test class " + testName, Project.MSG_VERBOSE); //$NON-NLS-1$
            if (isSuspicious(justClassName, testName)) {
                warnAbout(testName, testFile.getFullClassName(),
                        justClassName, importedClass);
                break;
            } else {
                log("The unit test " + testName + " seems to be well positioned.", Project.MSG_VERBOSE); //$NON-NLS-1$ //$NON-NLS-2$
            }
        }
    }

    private boolean isSuspicious(String justClassName, String testName) {
        boolean isSuspicious = false;
        for (String targetClassName : getPossibleClassUnderTestNames(testName)) {
            if (targetClassName.equals(justClassName)) {
                isSuspicious = true;
            }
        }
        return isSuspicious;
    }

    protected void warnAbout(String testClassName, String fullTestClassName,
            String unitClassName, String fullUnitClassName) {
        StringBuilder message = new StringBuilder();
        message.append("The unit test "); //$NON-NLS-1$
        message.append(testClassName);
        message.append(" seems to live in another package than its target class "); //$NON-NLS-1$
        message.append(unitClassName);
        message.append(":"); //$NON-NLS-1$
        log(message.toString(), Project.MSG_WARN);
        log("test class: " + fullTestClassName, Project.MSG_WARN); //$NON-NLS-1$
        log("unit class: " + fullUnitClassName, Project.MSG_WARN); //$NON-NLS-1$
    }

    private String[] getPossibleClassUnderTestNames(String testName) {
        return new String[] {
            getTestTargetClassNameFor(testName),
            //"I" + getTestTargetClassNameFor(testName), //$NON-NLS-1$
        };
    }

    protected String getTestTargetClassNameFor(String testClassName) {
        return testClassName.substring(0, testClassName.length() - "Test".length()); //$NON-NLS-1$
    }

    protected String getShortClassNameFrom(String fullClassName) {
        StringTokenizer tokenizer = new StringTokenizer(fullClassName, "."); //$NON-NLS-1$
        String token = null;
        while (tokenizer.hasMoreTokens()) {
            token = tokenizer.nextToken();
        }
        return token;
    }

    public static class FiletedClassFile {
        private final File classFile;
        private final String classFileEncoding;
        private String fullClassName;
        private List<String> importedClasses;

        public FiletedClassFile(File classFile, String encoding) {
            super();
            this.classFile = classFile;
            this.classFileEncoding = encoding;
            this.fullClassName = null;
            this.importedClasses = null;
        }

        public String getClassName() {
            return getClassNameFrom(this.classFile.getName());
        }

        public String getFullClassName() throws IOException {
            if (null == this.fullClassName) {
                parseClassSource();
            }
            return this.fullClassName;
        }

        public Iterable<String> getImportedClasses() throws IOException {
            if (null == this.importedClasses) {
                parseClassSource();
            }
            return this.importedClasses;
        }

        protected void parseClassSource() throws IOException {
            List<String> imports = new ArrayList<String>();
            BufferedReader reader = null;
            try {
                reader = new BufferedReader(
                            new InputStreamReader(
                                    new FileInputStream(this.classFile), this.classFileEncoding));
                String line = null;
                while (null != (line = reader.readLine())) {
                    String trimmedLine = line.trim();
                    if (isImportLine(trimmedLine)) {
                        imports.add(gatherClassNameFrom(trimmedLine));
                    }
                    if (isPackageLine(trimmedLine)) {
                        this.fullClassName = gatherPackageNameFrom(trimmedLine) + "." + getClassName(); //$NON-NLS-1$
                    }
                }
                this.importedClasses = imports;
            } finally {
                if (null != reader) {
                    reader.close();
                }
            }
        }

        protected String gatherClassNameFrom(String trimmedLine) {
            return trimmedLine.substring("import ".length(), trimmedLine.length() - 1); //$NON-NLS-1$
        }

        protected boolean isImportLine(String line) {
        	return (line.startsWith("import ") && line.endsWith(";")); //$NON-NLS-1$ //$NON-NLS-2$
        }

        protected String gatherPackageNameFrom(String trimmedLine) {
            return trimmedLine.substring("package ".length(), trimmedLine.length() - 1); //$NON-NLS-1$
        }

        protected boolean isPackageLine(String line) {
            return (line.startsWith("package ") && line.endsWith(";")); //$NON-NLS-1$ //$NON-NLS-2$
        }

        protected String getClassNameFrom(String fileName) {
            return fileName.replace(".java", ""); //$NON-NLS-1$ //$NON-NLS-2$
        }
    }
}
