package com.schneide.internal.anttask.helper;

import java.util.Collection;

import org.apache.tools.ant.BuildException;
import org.apache.tools.ant.Project;
import org.apache.tools.ant.Task;

/**
 * Abstract task base class that provides a lifecycle framework for parameter checking and
 * task execution. This class is capable to wrap custom exceptions in <code>BuildException<code>s,
 * so Ant can handle them.
 * 
 * @author dsl
 */
public abstract class AbstractAntTask extends Task {
    
    public AbstractAntTask() {
        super();
    }
    
    @Override
    public final void execute() throws BuildException {
        try {
            initialize();
            checkRequirements();
            executeHandled();
        } catch (Exception e) {
            if (e instanceof BuildException) {
                throw ((BuildException) e);
            }
            throw new BuildException(e);
        }
    }
    
    /**
     * Insert your task code here. If something goes wrong, you
     * should throw an exception.
     * 
     * @throws Exception if something got wrong. Will be handled by the superclass.
     */
    protected abstract void executeHandled() throws Exception;

    /**
     * Overwrite this method to check completeness and sanity of the given
     * parameters. You should consider using the failIfXYZ() methods.
     * Do not forget to call the super implementation, else you might miss checks.
     * 
     * @throws Exception if something got wrong. Will be handled by the superclass.
     */
    protected void checkRequirements() throws Exception {
        // nothing to check on this level
    }

    /**
     * Overwrite this method to perform tasks that are likely to fail (with an exception)
     * and therefor can't be done in the constructor.
     * 
     * @throws Exception if something got wrong. Will be handled by the superclass.
     */
    protected void initialize() throws Exception {
        // does nothing unless overwritten
    }
    
    protected final void failIfNull(Object parameter) throws BuildException {
        failIfNull(parameter, "");
    }
    
    protected final void failIfNull(Object parameter, String name) throws BuildException {
        if (null == parameter) {
            throw new BuildException("The required parameter " + name + " was not given.");
        }
    }
    
    protected final void failIfEmpty(Collection<?> collection) throws BuildException {
        if ((null == collection) || (collection.isEmpty())) {
            throw new BuildException("Missing collection parameter.");
        }
    }
    
    protected final void logVerbose(CharSequence text) {
        log(String.valueOf(text), Project.MSG_VERBOSE);
    }
}
