package com.schneide.internal.anttask;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.PrintWriter;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import org.apache.tools.ant.BuildException;
import org.apache.tools.ant.DirectoryScanner;
import org.apache.tools.ant.types.FileSet;

import com.schneide.internal.anttask.helper.AbstractAntTask;
import com.schneide.internal.anttask.model.IRelativeFileReference;

public class UnnecessaryCommentScanTask extends AbstractAntTask {

    private static final String[] FORBIDDEN_LINES = {
        " * To change the template for this generated file go to",
        " * Window>Preferences>Java>Code Generation>Code and Comments",
        " * TODO To change the template for this generated file go to",
        " * Window - Preferences - Java - Code Generation - Code and Comments",
        " * Window - Preferences - Java - Code Style - Code Templates",
        " * TODO To change the template for this generated type comment go to",
        " * Window&gt;Preferences&gt;Java&gt;Code Generation&gt;Code and Comments",
        " * To change the template for this generated type comment go to",
    };

    private final List<FileSet> filesets;
    private boolean failOnOccurrences;
    private File reportFile;
    private boolean isVerbose;

    public UnnecessaryCommentScanTask() {
        super();
        this.filesets = new ArrayList<FileSet>();
        this.failOnOccurrences = false;
        this.reportFile = null;
        this.isVerbose = true;
    }

    public void addConfigured(FileSet files) {
        this.filesets.add(files);
    }

    public void setFailOnOccurrences(boolean failOnOccurrences) {
        this.failOnOccurrences = failOnOccurrences;
    }

    public void setReportFile(File reportFile) {
        this.reportFile = reportFile;
    }

    public void setVerbose(boolean verbose) {
        this.isVerbose = verbose;
    }

    @Override
    protected void checkRequirements() throws Exception {
        super.checkRequirements();
        failIfEmpty(this.filesets);
        failIfNull(this.reportFile);
    }

    @Override
    protected void executeHandled() throws Exception {
        Iterable<IRelativeFileReference> includedFiles = getIncludedFiles();
        int scannedFileCount = 0;
        List<Finding> findings = new ArrayList<Finding>();
        for (IRelativeFileReference currentFile : includedFiles) {
            Iterable<Finding> currentFindings = scanFile(currentFile);
            for (Finding finding : currentFindings) {
                findings.add(finding);
            }
            scannedFileCount++;
        }
        logResults(scannedFileCount, findings.toArray(new Finding[findings.size()]));
        if (this.failOnOccurrences && (!findings.isEmpty())) {
            throw new BuildException("Found invalid comment lines. Please review the finding log.", getLocation());
        }
    }

    private void logResults(int scannedFilesCount, Finding... findings) throws Exception {
        if (null != this.reportFile) {
            new FileFindingLogger(this.reportFile).logFindings(scannedFilesCount, findings);
        }
        if (this.isVerbose) {
            new BuildlogFindingLogger().logFindings(scannedFilesCount, findings);
        }
    }

    private class BuildlogFindingLogger implements FindingLogger {
        public BuildlogFindingLogger() {
            super();
        }
        @Override
        public void logFindings(int scannedFilesCount, Finding... findings) throws Exception {
            log("Scanned " + scannedFilesCount + " files and found " + findings.length + " unnecessary comments.");
            for (Finding finding : findings) {
                log("Found unnecessary comment at line " + finding.getLineNumber() + " in file " + finding.getFile().getRelativePath() + ": " + finding.getLine());
            }
        }
    }

    private class FileFindingLogger implements FindingLogger {
        private final File targetFile;
        public FileFindingLogger(File targetFile) {
            super();
            this.targetFile = targetFile;
        }
        @Override
        public void logFindings(int scannedFilesCount, Finding... findings) throws Exception {
            PrintWriter writer = new PrintWriter(new FileWriter(this.targetFile));
            writer.println("Scanned " + scannedFilesCount + " files and found " + findings.length + " unnecessary comments.");
            for (Finding finding : findings) {
                writer.println("Found unnecessary comment at line " + finding.getLineNumber() + " in file " + finding.getFile().getRelativePath() + ": " + finding.getLine());
            }
            writer.flush();
            writer.close();
        }
    }

    private static class Finding {
        private final IRelativeFileReference file;
        private final int lineNumber;
        private final String line;

        public Finding(IRelativeFileReference file, int lineNumber, String line) {
            super();
            this.file = file;
            this.lineNumber = lineNumber;
            this.line = line;
        }

        public IRelativeFileReference getFile() {
            return file;
        }

        public String getLine() {
            return line;
        }

        public int getLineNumber() {
            return lineNumber;
        }
    }

    private interface FindingLogger {
        public void logFindings(int scannedFilesCount, Finding... findings) throws Exception;
    }

    private Iterable<Finding> scanFile(IRelativeFileReference file) throws IOException {
        List<Finding> result = new ArrayList<Finding>();
        BufferedReader reader = new BufferedReader(new FileReader(file.getFile()));
        try {
	        String line = null;
	        int lineCount = 0;
	        while ((line = reader.readLine()) != null) {
	            if (isForbiddenLine(line)) {
	                result.add(new Finding(file, lineCount, line));
	            }
	            lineCount++;
	        }
        } finally {
        	reader.close();
        }
        return result;
    }

    private Iterable<IRelativeFileReference> getIncludedFiles() {
        List<IRelativeFileReference> result = new ArrayList<IRelativeFileReference>();
        for (FileSet fileset : this.filesets) {
            Collections.addAll(result, getIncludedFilesOf(fileset));
        }
        return result;
    }

    private IRelativeFileReference[] getIncludedFilesOf(FileSet fileset) {
        DirectoryScanner directoryScanner = fileset.getDirectoryScanner(getProject());
        directoryScanner.scan();
        String[] relativePaths = directoryScanner.getIncludedFiles();
        List<IRelativeFileReference> result = new ArrayList<IRelativeFileReference>();
        for (String path : relativePaths) {
            result.add(new RelativeFileReference(
                    new File(directoryScanner.getBasedir(), path),
                    directoryScanner.getBasedir(),
                    path));
        }
        return result.toArray(new IRelativeFileReference[result.size()]);
    }

    private static class RelativeFileReference implements IRelativeFileReference {
        private final File file;
        private final File baseDirectory;
        private final String relativePath;

        public RelativeFileReference(File file,
                File baseDirectory,
                String relativePath) {
            super();
            this.file = file;
            this.baseDirectory = baseDirectory;
            this.relativePath = relativePath;
        }

        public File getBaseDirectory() {
            return this.baseDirectory;
        }

        public File getFile() {
            return this.file;
        }

        public String getRelativePath() {
            return this.relativePath;
        }
    }

    private boolean isForbiddenLine(String line) {
        for (int i = 0; i < FORBIDDEN_LINES.length; i++) {
            if (FORBIDDEN_LINES[i].equals(line)) {
                return true;
            }
        }
        return false;
    }
}
