package com.schneide.internal.anttask;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.jar.JarEntry;
import java.util.jar.JarInputStream;

import org.apache.tools.ant.BuildException;
import org.apache.tools.ant.Project;
import org.apache.tools.ant.Task;
import org.apache.tools.ant.taskdefs.condition.Condition;
import org.apache.tools.ant.types.Path;

/**
 * Checks a given classpath for resources with the same relative URL.<br/>
 * If such a collision is detected, it may be logged or the build aborted.<br/>
 * If used as a condition, it will result in <code>true</code> if no collisions appeared.
 *
 * @author dali
 */
public class ClasspathCollisionCheckTask extends Task implements Condition {

    private final Path classpath;
    private boolean onlyClassFiles;
    private boolean isVerbose;
    private boolean failOnCollision;

    public ClasspathCollisionCheckTask() {
        super();
        this.classpath = new Path(getProject());
        this.onlyClassFiles = true;
        this.isVerbose = false;
        this.failOnCollision = false;
    }

    public void addConfigured(Path classpath) {
        this.classpath.add(classpath);
    }

    public void setOnlyClasses(boolean isEnabled) {
        this.onlyClassFiles = isEnabled;
    }

    public boolean considerOnlyClasses() {
        return this.onlyClassFiles;
    }

    public void setVerbose(boolean isEnabled) {
        this.isVerbose = isEnabled;
    }

    public boolean isVerbose() {
        return this.isVerbose;
    }

    public void setFailOnCollision(boolean isEnabled) {
        this.failOnCollision = isEnabled;
    }

    public boolean isFailingOnCollision() {
        return this.failOnCollision;
    }

    public boolean eval() throws BuildException {
        return (!hasCollisions(determineCollisions()));
    }

    private boolean hasCollisions(Collision[] collisions) {
        return (collisions.length > 0);
    }

    @Override
    public void execute() throws BuildException {
        Collision[] collisions = determineCollisions();
        log(getCollisionSummary(collisions));
        if (isVerbose()) {
            for (Collision collision : collisions) {
                logVerbose(String.valueOf(collision));
            }
        }
        if (isFailingOnCollision() && hasCollisions(collisions)) {
            throw new BuildException(getCollisionSummary(collisions));
        }
    }

    private String getCollisionSummary(Collision[] collisions) {
        StringBuilder result = new StringBuilder();
        result.append("Detected "); //$NON-NLS-1$
        if (!hasCollisions(collisions)) {
            result.append("no"); //$NON-NLS-1$
        } else {
            result.append(String.valueOf(collisions.length));
        }
        result.append(" classpath collisions."); //$NON-NLS-1$
        return result.toString();
    }

    private Collision[] determineCollisions() {
        List<Collision> result = new ArrayList<Collision>();
        Map<String, File> globalElements = new HashMap<String, File>();
        for (String string : this.classpath.list()) {
            File pathFile = new File(string);
            if (!pathFile.isFile()) {
                continue;
            }
            String[] elements = getElementsFor(pathFile);
            for (String element : elements) {
                if (element.endsWith("/")) { //$NON-NLS-1$
                    continue;
                }
                if (considerOnlyClasses() && (!element.endsWith(".class"))) { //$NON-NLS-1$
                    continue;
                }
                if (globalElements.containsKey(element)) {
                    File otherFile = globalElements.get(element);
                    Collision collision = new Collision(element, otherFile,
                            pathFile);
                    result.add(collision);
                } else {
                    globalElements.put(element, pathFile);
                }
            }
        }

        logVerbose("Detected " + globalElements.size() + " distinguishable classpath elements."); //$NON-NLS-1$ //$NON-NLS-2$
        return result.toArray(new Collision[result.size()]);
    }

    private void logVerbose(CharSequence text) {
        if (isVerbose()) {
            log(String.valueOf(text));
        }
    }

    private static class Collision {
        private final String resourcePath;
        private final File firstFile;
        private final File secondFile;

        public Collision(String resourcePath, File firstFile, File secondFile) {
            super();
            this.resourcePath = resourcePath;
            this.firstFile = firstFile;
            this.secondFile = secondFile;
        }

        public File getFirstFile() {
            return this.firstFile;
        }

        public File getSecondFile() {
            return this.secondFile;
        }

        public String getResourcePath() {
            return this.resourcePath;
        }

        @Override
        public String toString() {
            StringBuilder result = new StringBuilder();
            result.append(getResourcePath());
            result.append(" exists in both "); //$NON-NLS-1$
            result.append(getFirstFile());
            result.append(" and "); //$NON-NLS-1$
            result.append(getSecondFile());
            return result.toString();
        }
    }

    private String[] getElementsFor(File jarFile) {
        List<String> result = new ArrayList<String>();
        FileInputStream fileInput = null;
        JarInputStream jarInput = null;
        try {
            fileInput = new FileInputStream(jarFile);
            jarInput = new JarInputStream(fileInput);
            JarEntry currentEntry = null;
            while (null != (currentEntry = jarInput.getNextJarEntry())) {
                result.add(currentEntry.getName());
            }
        } catch (IOException e) {
        	getProject().log("error while reading jar file content " + jarFile, e, Project.MSG_ERR);
        } finally {
            closeStream(jarInput);
            closeStream(fileInput);
        }
        return result.toArray(new String[result.size()]);
    }

    private void closeStream(InputStream stream) {
        try {
            if (null != stream) {
                stream.close();
            }
        } catch (IOException e) {
        	getProject().log("error while closing input stream.", e, Project.MSG_ERR);
        }
    }
}
