#include <catch2/catch.hpp>
#include "vera_base/Spectrum.hpp"

using namespace Catch::literals;
using namespace vera;

TEST_CASE("Subtract spectrums")
{
  auto a = Spectrum(Interval::unit(), {10.0}) - Spectrum(Interval::unit(), {4.0});
  REQUIRE(a.intensities()[0] == 6.0_a);
}

TEST_CASE("Log spectrums")
{
  auto a = logOrZero(Spectrum(Interval::unit(), {std::exp(1.0)}));
  REQUIRE(a.intensities()[0] == 1.0_a);
}

TEST_CASE("Finds relative peak")
{
  Spectrum sample({-3.0, 3.0}, {1.0, 0.0, 3.0, 6.0, 3.0, 2.0, 4.0});

  auto info = PeakInfo::computeFrom(sample, -1.0, 2.0, 1.0);
  REQUIRE(info.relativePeak() == 5.0_a); // 6.0 over a base line with 1.0 at the center
}

TEST_CASE("Constructs from line")
{
  Spectrum sample({{3.0, 6.0}, {1.0, 5.0}}, 5);
  REQUIRE(sample.intensities().size() == 5);
  REQUIRE(sample.intensities()[0] == 1.0_a);
  REQUIRE(sample.intensities()[1] == 2.0_a);
  REQUIRE(sample.intensities()[2] == 3.0_a);
  REQUIRE(sample.intensities()[3] == 4.0_a);
  REQUIRE(sample.intensities()[4] == 5.0_a);
}

TEST_CASE("Smoothing does not depend on number of points")
{
  Spectrum sample({-12.0, 12.0},  {0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 1.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0});
  Spectrum sample2({-10.0, 10.0}, {0.0, 0.0, 0.0, 0.0, 0.0, 1.0, 0.0, 0.0, 0.0, 0.0, 0.0});
  auto smoothed = gauss(sample, 2.0);
  auto smoothed2 = gauss(sample2, 2.0);

  auto const& values = smoothed.intensities();
  auto center = std::vector<double>(values.begin()+1, values.begin()+values.size()-1);
  REQUIRE(center == smoothed2.intensities());
}