package com.schneide;

import org.apache.cassandra.db.Mutation;
import org.apache.cassandra.db.partitions.Partition;
import org.apache.cassandra.db.rows.Row;
import org.apache.cassandra.db.rows.Unfiltered;
import org.apache.cassandra.db.rows.UnfilteredRowIterator;
import org.apache.cassandra.triggers.ITrigger;

import java.io.IOException;
import java.net.*;
import java.nio.charset.StandardCharsets;
import java.util.Collection;
import java.util.Collections;
import java.util.Optional;
import java.util.Properties;
import java.util.concurrent.LinkedBlockingDeque;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

public class UdpChanges implements ITrigger {

    private DatagramSocket socket;
    private InetAddress address;
    private short port;

    public UdpChanges() {
        try {
            socket = new DatagramSocket();
            address = InetAddress.getByName(Optional.ofNullable(System.getenv("UDP_CHANGES_HOST")).orElse("localhost"));
            port = Optional.ofNullable(System.getenv("UDP_CHANGES_PORT")).map(Short::parseShort).orElse((short)30001);
        } catch (SocketException | UnknownHostException e) {
            e.printStackTrace();
        }
    }
    private void send(final String message) {
        final byte[] buffer = message.getBytes(StandardCharsets.UTF_8);
        final DatagramPacket packet
                = new DatagramPacket(buffer, buffer.length, address, port);
        try {
            socket.send(packet);
        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    private void report(final String message) {
        send(message);
    }

    @Override
    public Collection<Mutation> augment(Partition partition) {
        if (partition.partitionLevelDeletion().isLive()) {
            UnfilteredRowIterator it = partition.unfilteredIterator();
            while (it.hasNext()) {
                final Unfiltered un = it.next();
                switch (un.kind()) {
                    case ROW:
                        // row
                        Row row = (Row) un;
                        if (row.primaryKeyLivenessInfo().timestamp() != Long.MIN_VALUE) {
                            // row insert
                            // only INSERT operation updates row timestamp (LivenessInfo).
                            // For other operations this timestamp is not updated and equals Long.MIN_VALUE
                            report("row_insert");

                        } else {
                            if (row.deletion().isLive()) {
                                report("row_update");
                            }
                        }
                        break;
                    case RANGE_TOMBSTONE_MARKER:
                        report("range_deletion");
                        break;
                }
            }
        } else {
            report("partition_delete");
        }
        return Collections.emptyList();
    }
}
